from datetime import datetime, time
from django.db.models.signals import post_migrate
from django.dispatch import receiver
from django.utils import timezone
from .models import BusinessHours, Service


@receiver(post_migrate)
def create_default_business_hours(sender, **kwargs):
    """
    Create default business hours (9 AM - 5 PM, Monday to Friday) 
    when the app is first installed.
    """
    if sender.name == 'appointments':
        # Days of week: 0=Monday, 1=Tuesday, ..., 6=Sunday
        days = [
            (0, False, time(9, 0), time(17, 0)),  # Monday - open
            (1, False, time(9, 0), time(17, 0)),  # Tuesday - open
            (2, False, time(9, 0), time(17, 0)),  # Wednesday - open
            (3, False, time(9, 0), time(17, 0)),  # Thursday - open
            (4, False, time(9, 0), time(17, 0)),  # Friday - open
            (5, True, time(0, 0), time(0, 0)),    # Saturday - closed (using 00:00 as placeholder)
            (6, True, time(0, 0), time(0, 0)),    # Sunday - closed (using 00:00 as placeholder)
        ]
        
        for day_num, is_closed, open_time, close_time in days:
            # Only set open_time and close_time if the business is open that day
            if is_closed:
                BusinessHours.objects.get_or_create(
                    day_of_week=day_num,
                    defaults={
                        'is_closed': True,
                        'open_time': time(0, 0),  # Use midnight as placeholder
                        'close_time': time(0, 0),  # Use midnight as placeholder
                    }
                )
            else:
                BusinessHours.objects.get_or_create(
                    day_of_week=day_num,
                    defaults={
                        'is_closed': False,
                        'open_time': open_time,
                        'close_time': close_time,
                    }
                )


@receiver(post_migrate)
def create_default_services(sender, **kwargs):
    """
    Create some default services when the app is first installed.
    """
    if sender.name == 'appointments':
        default_services = [
            {
                'name': 'Haircut',
                'description': 'Professional haircut service',
                'duration': timezone.timedelta(minutes=30),
                'price': 25.00,
                'is_active': True
            },
            {
                'name': 'Hair Coloring',
                'description': 'Full hair coloring service',
                'duration': timezone.timedelta(hours=2),
                'price': 80.00,
                'is_active': True
            },
            {
                'name': 'Manicure',
                'description': 'Basic manicure service',
                'duration': timezone.timedelta(minutes=45),
                'price': 20.00,
                'is_active': True
            },
            {
                'name': 'Pedicure',
                'description': 'Basic pedicure service',
                'duration': timezone.timedelta(hours=1),
                'price': 30.00,
                'is_active': True
            },
            {
                'name': 'Facial',
                'description': 'Basic facial treatment',
                'duration': timezone.timedelta(minutes=50),
                'price': 45.00,
                'is_active': True
            },
        ]
        
        for service_data in default_services:
            Service.objects.get_or_create(
                name=service_data['name'],
                defaults=service_data
            )
